<?php
/************************************************************************
 * Domain Analyzer PHP Class v1.1
 ************************************************************************
 * Domain Analyzer php class to analyze domain for different stats
 * like: alexa rank, google page rank, etc etc.
 ************************************************************************
 * @package     class.analyzer.1.1
 * @link        http://websterfolks.com/forums/php-classes.148/
 * @updated     27/03/2013
 * @author      Mandeep Singh <hello@mesingh.com>
 * @copyright   Copyright (c) 2013 WebsterFolks, All rights reserved
 * @license     http://codecanyon.net/licenses
 ***********************************************************************/
class Analyzer
{

    /**
     * Object for url
     *
     * @access        public
     * @var           string
     */
    public $url;

    /**
     * Temp variable
     */
    public $temp;

    /**
     * Track the instance for google page rank
     *
     * @access        private
     * @var           string
     */
    private static $instance;

    /**
     * Constructor
     *
     * Initialize the url provided by user.
     *
     * @access        public
     * @param         string        $url        String, containing the initialized
     *                                          object URL.
     */
    public function __construct($temp)
    {
        $temp = str_replace(' ', '', $temp);
        $this->url = $temp;
    }

    /**
     * Validates the initialized object URL syntax.
     *
     * @access        public
     * @param         string        $url        String, containing the initialized object URL.
     * @return        string                    Returns string, containing the validation result.
     */
    public function valid_url()
    {
        if(!isset($this->url) || $this->url == '')
        {
            $e = 'Invalid Object > Requires an URL.';
        }
        else
        {
            $pattern  = '/^(https?:\/\/)?([\da-z\.-]+)\.([a-z\.]{2,6})([\/\w \.-]*)*\/?$/';
            if(!preg_match($pattern, $this->url))
            {
                $e = 'Invalid URL > Invalid URL syntax.';
            }
            else
            {
                $allowed_schemes = array('http','https');
                $host     = parse_url($this->url, PHP_URL_HOST);
                $scheme = parse_url($this->url, PHP_URL_SCHEME);
                if(!in_array(strtolower($scheme),$allowed_schemes))
                {
                    $this->url = 'http://'.$this->url;
                    $e = 'valid';
                }
                elseif(empty($host) || $host == '')
                {
                    $e = 'Invalid URL > Hostname undefined (or invalid URL syntax).';
                }
                else
                {
                    $e = 'valid';
                }
            }
        }
        return $e;
    }

    /**
     * Get the data from social api`s using cURL
     *
     * @access        private
     * @param         string        $api_url        String, containing the api URL.
     * @return        string                        Returns string, containing the content fetched from api.
     */
    private function use_curl($api_url)
    {   
        $ch  = curl_init($api_url);
        curl_setopt($ch,CURLOPT_RETURNTRANSFER,1);
        curl_setopt($ch,CURLOPT_CONNECTTIMEOUT,5);
        curl_setopt($ch,CURLOPT_FOLLOWLOCATION,1);
        curl_setopt($ch,CURLOPT_MAXREDIRS,2);
        if(strtolower(parse_url($api_url, PHP_URL_SCHEME)) == 'https')
        {
            curl_setopt($ch,CURLOPT_SSL_VERIFYPEER,1);
            curl_setopt($ch,CURLOPT_SSL_VERIFYHOST,1);
        }
        $content = curl_exec($ch);
        curl_close($ch);
        return $content;
    }


    /**
     * Get the data from various api`s.
     *
     * @access        private
     * @param         string        $api_url        String, containing the api URL.
     * @return        string                    	Returns string, containing the content fetched from api.
     */
    private function get_content($api_url)
    {
        if(in_array('curl', get_loaded_extensions()))
        {
            $content = self::use_curl($api_url);
        }
        else
        {
            $content = file_get_contents($api_url);
        }
    	return $content;
    }

    /**
     * Check if domain is present in dmoz or not
     *
     * @access        public
     * @return        boolean                        Returns boolean, true if site is present 
     *                                               in dmoz else false
     */
    public function get_dmoz() 
    {
        $api_url ="http://search.dmoz.org/cgi-bin/search?search=".$this->url;
        $content = self::get_content($api_url);
        if (empty($content)) 
        {
            return intval(0);
        }
        return (strpos($content, "Open Directory Categories") && strpos($content,"Open Directory Sites"))
            ? true : false ;
    }

    /**
     * Check if domain is present in yahoo directory or not
     *
     * @access        public
     * @return        boolean                        Returns boolean, true if site is present 
     *                                               in yahoo directory else false
     */
    public function get_yahoo_directory() 
    {
        $api_url = "http://search.yahoo.com/search/dir?p=".$this->url;
        $content = self::get_content($api_url);
        if (empty($content)) 
        {
            return intval(0);
        }
        return strpos($content, "We did not find results for:") ? false : true;
    }

    /**
     * Get the backlinks count in google
     *
     * @access        public
     * @return        integer                        Returns integer, backlink count in google
     */ 
    public function get_google_backlinks() 
    {
        $api_url = "http://www.google.com/search?q=link%3A".$this->url;
        $content = self::get_content($api_url);
        if (empty($content)) 
        {
            return intval(0);
        }
        if (!strpos($content, 'results')) return intval(0);
        $match_expression = '/About (.*?) results/sim'; 
        preg_match($match_expression,$content,$matches); 
        if (empty($matches)) return intval(0);
        return intval(str_replace(",", "", $matches[1]));       
    }

    /**
     * Get the backlinks count in bing
     *
     * @access        public
     * @return        integer                        Returns integer, backlink count in bing
     */
    public function get_bing_backlinks(){ 
        $api_url = "http://www.bing.com/search?q=link%3A".$this->url;
        $content = self::get_content($api_url);
        if (empty($content)) 
        {
            return intval(0);
        }
        if (!strpos($content, '<span class="sb_count" id="count">')) return intval(0);
        
        $match_expression = '/<span class="sb_count" id="count">(.*?)<\\/span>/im';
        preg_match_all($match_expression, $content, $matches);

        $data = str_replace(",", "", $matches[0]);
        $data = str_replace('<span class="sb_count" id="count">', "", $data);
        $data = str_replace("</span>", "", $data);

        return intval(substr($data[0], 0, -8));
    }

    /**
     * Get the google indexed pages count
     *
     * @access        public
     * @return        integer                        Returns integer, indexed pages count in google
     */
    public function get_google_indexed() {
        $api_url = "http://www.google.com/search?q=site%3A".$this->url;
        $content = self::get_content($api_url);
        if (empty($content)) 
        {
            return intval(0);
        }
        if (!strpos($content, 'results')) return intval(0);
        $match_expression = '/About (.*?) results/sim'; 
        preg_match($match_expression,$content,$matches); 
        if (empty($matches)) return intval(0);
        return intval(str_replace(",", "", $matches[1]));
    }

    /**
     * Get the alexa rank for doamin
     *
     * @access        public
     * @return        integer                        Returns integer, alexa rank for domain
     */
    public function get_alexa_rank(){
        $api_url = "http://data.alexa.com/data?cli=10&dat=snbamz&url=".$this->url;
        $content = self::get_content($api_url);
        if (empty($content)) 
        {
            return intval(0);
        }
        else
        {
            $xml = new SimpleXMLElement($content);  
            $popularity = $xml->xpath("//POPULARITY");
            if(isset($popularity[0]['TEXT']))
            {
                $rank = (string)$popularity[0]['TEXT']; 
            }
            else
            {
                $rank = 0;
            }
            return intval($rank);
        }
    }

    /**
     * Get google page rank for doamin
     *
     * @access        public
     * @return        integer                        Returns integer, google page rank for domain
     */
    public function get_google_page_rank() {
        $analyzer_google_rank = new Analyzer_Google_Rank();
        return intval($analyzer_google_rank->get_rank($this->url));
    }


    /**
     * Get all stats in one go 
     *
     * @access          public
     * @return          array                     Returns array of all stats.
     */
    public function get_all(){
        $data = array();
        $data['dmoz'] = self::get_dmoz();
        $data['yahoo_directory'] = self::get_yahoo_directory();
        $data['google_backlinks'] = self::get_google_backlinks();
        $data['bing_backlinks'] = self::get_bing_backlinks();
        $data['google_indexed'] = self::get_google_indexed();
        $data['alexa_rank'] = self::get_alexa_rank();
        $data['google_page_rank'] = self::get_google_page_rank();

        return $data;
    }

}

class Analyzer_Google_Rank
{

    /**
     * Track the instance
     *
     * @access        private
     * @var           string
     */
    private static $instance;

    /**
     * Get google page rank for doamin
     *
     * @access        public
     * @return        integer                        Returns integer, google page rank for domain
     */
    public function get_rank($url) 
    {
        if(!isset(self::$instance)) 
        {
          self::$instance = new self();
        }
        return self::$instance->check($url);
    }

    /**
     * Helper function for google page rank check
     */
    function stringToNumber($string,$check,$magic) 
    {
        $int32 = 4294967296;
        $length = strlen($string);
        for ($i = 0; $i < $length; $i++) 
        {
            $check *= $magic;
            if($check >= $int32) {
            $check = ($check - $int32 * (int) ($check / $int32));
            $check = ($check < -($int32 / 2)) ? ($check + $int32) : $check;
            }
            $check += ord($string{$i});
        }
        return $check;
    }

    /**
     * Helper function for google page rank check
     */
    function createHash($string) 
    {
        $check1 = $this->stringToNumber($string, 0x1505, 0x21);
        $check2 = $this->stringToNumber($string, 0, 0x1003F);

        $factor = 4;
        $halfFactor = $factor/2;

        $check1 >>= $halfFactor;
        $check1 = (($check1 >> $factor) & 0x3FFFFC0 ) | ($check1 & 0x3F);
        $check1 = (($check1 >> $factor) & 0x3FFC00 ) | ($check1 & 0x3FF);
        $check1 = (($check1 >> $factor) & 0x3C000 ) | ($check1 & 0x3FFF);  

        $calc1 = (((($check1 & 0x3C0) << $factor) | ($check1 & 0x3C)) << $halfFactor ) | ($check2 & 0xF0F );
        $calc2 = (((($check1 & 0xFFFFC000) << $factor) | ($check1 & 0x3C00)) << 0xA) | ($check2 & 0xF0F0000 );
        return ($calc1 | $calc2);
    }

    /**
     * Helper function for google page rank check
     */
    function checkHash($hashNumber)
    {
        $check = 0;
        $flag = 0;
        $hashString = sprintf('%u', $hashNumber) ;
        $length = strlen($hashString);
        for ($i = $length - 1;  $i >= 0;  $i --) 
        {
            $r = $hashString{$i};
            if(1 === ($flag % 2)) 
            {
                $r += $r;
                $r = (int)($r / 10) + ($r % 10);
            }
            $check += $r;
            $flag ++;
        }
        $check %= 10;
        if(0 !== $check) 
        {
            $check = 10 - $check;
            if(1 === ($flag % 2) ) 
            {
                if(1 === ($check % 2)) 
                {
                    $check += 9;
                }
                $check >>= 1;
            }
        }
        return '7'.$check.$hashString;
    }

    /**
     * Helper function for google page rank check
     */
    function check($page) 
    {
        $socket = fsockopen("toolbarqueries.google.com", 80, $errno, $errstr, 30);
        if($socket) 
        {
            $out = "GET /tbr?client=navclient-auto&ch=".$this->checkHash($this->createHash($page)).
            "&features=Rank&q=info:".$page."&num=100&filter=0 HTTP/1.1\r\n";
            $out .= "Host: toolbarqueries.google.com\r\n";
            $out .= "User-Agent: Mozilla/4.0 (compatible; GoogleToolbar 2.0.114-big; Windows XP 5.1)\r\n";
            $out .= "Connection: Close\r\n\r\n";
            fwrite($socket, $out);
            $result = "";
            while(!feof($socket)) 
            {
                $data = fgets($socket, 128);
                $pos = strpos($data, "Rank_");
                if($pos !== false)
                {
                    $pagerank = substr($data, $pos + 9);
                    $result += $pagerank;
                }
            }
            fclose($socket);
            return $result;
        }
    }
}